/**
 * \file: pfcfg_interface.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * Interface for Platform Configuration (pfcfg).
 * Interface for reading and writing configuration data for linux userspace a
 * applications
 * \component: pfcfg
 *
 * \author: M. Methner
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \history
 * 0.1 M. Methner Initial version
 *
 ***********************************************************************/

#ifndef _PFCFG_INTERFACE_H_
#define _PFCFG_INTERFACE_H_

#include <adit_typedef.h>

#ifdef  __cplusplus
extern "C" {
#endif

#define MAX_CFG_LINE_LENGTH 2048


typedef char PFCFG_CHAR;

typedef enum
{
    PFCFG_NO_ERROR = 0,
    PFCFG_COMMON_ERROR = -100,
    PFCFG_CONFIG_NOT_FOUND,
    PFCFG_KEY_NOT_FOUND,
    PFCFG_NOT_IMPLEMENTED
} PFCFG_ERR;

typedef enum
{
    PFCFG_READ = 0,
    PFCFG_READWRITE
} PFCFG_MODE;

typedef struct PFCFG_Handle_t PFCFG_Handle_t;


/**
* \func PFCFG_opn_cnf
*
* Open configuration dataset.
*
*
* \param p_name Name of configuration set to open
* \param mode open configuration set in read or read/write mode
* 
*
* \return handle for the configration set or null in error case
*
*/
PFCFG_Handle_t *
PFCFG_opn_cnf(const PFCFG_CHAR * p_name, PFCFG_MODE mode);

/**
* \func PFCFG_close_cnf
*
* Close configuration set.
*
*
* \param p_handle handle for the configration set to close
*
* \return PFCFG_NO_ERROR
*
*/
PFCFG_ERR
PFCFG_close_cnf(PFCFG_Handle_t * p_handle);


/**
* \func PFCFG_get_cfs
*
* Get configuration string.
*
* \param p_handle handle for the configuration set
* \param p_key key name to read from
* \param p_str string to store read value
* \param length sizeof(p_str)
*
* \return PFCFG_KEY_NOT_FOUND or PFCFG_NO_ERROR
*
*/
PFCFG_ERR
PFCFG_get_cfs(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, PFCFG_CHAR * p_str, U32 length);


/**
* \func PFCFG_get_cfn
*
* Get configuration integers.
*
* \param handle handle for the configuration set
* \param p_key key name to read from
* \param p_values pointer to integer array to store values
* \param length maximum number of values to store
*
* \return PFCFG_KEY_NOT_FOUND or number of integers which a stored in
* the configuration set
*
*/
PFCFG_ERR
PFCFG_get_cfn(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, S32 * p_values, U32 length);


/**
* \func PFCFG_get_cfb
*
* Get configuration binary object (not implemented yet).
*
* \param id ID of configuration set
* \param p_key key name to read from
* \param p_data pointer to store the data
* \param length maximum number of bytes to store
*
* \return PFCFG_NOT_IMPLEMENTED
*
*/
PFCFG_ERR
PFCFG_get_cfb(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, VP * p_data, U32 length);


/**
* \func PFCFG_set_cfs
*
* Set configuration string (not implemented yet).
*
* \param id ID of configuration set
* \param p_key key name to write to
* \param p_str string to write
* \param length length of the string
*
* \return PFCFG_NOT_IMPLEMENTED
*
*/
PFCFG_ERR
PFCFG_set_cfs(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, const PFCFG_CHAR * p_str, U32 length);


/**
* \func PFCFG_set_cfn
*
* set configuration integers (not implemented yet).
*
* \param id ID of configuration set
* \param p_key key name to write to
* \param p_values pointer to integer array to store
* \param length number of values to store
*
* \return PFCFG_NOT_IMPLEMENTED
*
*/
PFCFG_ERR
PFCFG_set_cfn(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, const S32 * p_values, U32 length);


/**
* \func PFCFG_set_cfb
*
* set binary data (not implemented yet).
*
* \param id ID of configuration set
* \param p_key key name to write to
* \param p_data Data to store
* \param length number of bytes
*
* \return PFCFG_NOT_IMPLEMENTED
*
*/
PFCFG_ERR
PFCFG_set_cfb(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, const VP * p_data, U32 length);

#ifdef  __cplusplus
}
#endif


#endif
